// domain data encoding/decoding algo for FrameworkPOS Malware DNS-Tunneling Variant, 
// as described on: 
// https://blog.gdata.de/artikel/neue-variante-von-frameworkpos-schoepft-daten-ueber-dns-anfragen-ab/
//
package main

import(
	"fmt"
	"os"
	"encoding/hex"
	"flag"
)

// test vectors:
// data := []byte{0xc3, 0xcb, 0xc0, 0xdc, 0xc3, 0xc4, 0xca, 0xdc, 0xc4, 0xcb, 0xdc, 0xc4, 0xcb}
// string := "192.168.69.69"
//
// data := []byte{0xa2, 0xb3, 0xa7, 0xbe, 0xdf, 0xb3, 0xb0, 0xb1, 0xc3, 0xc0, 0xc1, 0xc6}
// string := "PAUL-ABC1234"
const A = 0xAA
const B = 0x9B
const C = 0xC3

var op string
var in string

func init(){
	flag.StringVar(&op, "op", "decode", "operation to perform: 'encode' or 'decode', default is 'decode'")
	flag.StringVar(&in, "in", "", "Inputstring to perform choosen operation on")
}

// decode - decodes single values from dns queries into cleartext
func decode(data []byte)(result []byte){
	for _, v := range data {
		a := v ^ A
		b := a ^ B
		result = append(result, b ^ C)
	}
	return result
}

// encode - encodes cleartext values to be used in dns queries
func encode(data []byte)(result []byte) {
	for _, v := range data {
		b := v ^ C
		a := b ^ B
		result = append(result, a ^ A)
	}
	return result
}

func main(){
	flag.Parse()
	input := []byte(in)

	switch {
	case op == "encode":
		encoded := encode(input)
		fmt.Printf("%s %x\n", input, string(encoded))
	case op == "decode":
		data, err := hex.DecodeString(string(input))
		if err != nil {
			panic(err)
		}
		decoded := decode(data)
		fmt.Printf("%x %s\n", data, string(decoded))
	default:
		prog := os.Args[0]
		fmt.Printf("For USAGE INFO call: '%s -h'\n", prog)
	}
}