# Protecting sshd with spiped

This repository shows how to place **spiped** in front of your SSH daemon (`sshd`) in order to add an extra layer of symmetric-key protection and drastically reduce your exposed attack surface.

---

## What is spiped?

`spiped` is a lightweight daemon designed to create symmetrically encrypted and authenticated “pipes” between socket addresses. It binds a local TCP port and forwards traffic to another socket—typically on the same machine—**while encrypting and verifying** all data in transit.

It resembles the port-forwarding feature of SSH (`ssh -L`), but uses a pre-shared symmetric key rather than SSH’s full session model, making it simpler and more focused on **secure tunneling**.

---

## Why protect your SSH daemon with spiped?

Running `sshd` behind a `spiped` tunnel adds a strong additional security layer that prevents nearly all unauthorized SSH access attempts before they even reach your server.

Because `spiped` only accepts connections encrypted with a pre-shared key, any unauthenticated traffic is dropped immediately—long before it can trigger password guessing, exploit attempts, or even banner visibility.

### Key advantages

- **Eliminates SSH port scanning & brute-force noise**  
  Scanners and bots see nothing: your SSH port becomes completely inaccessible unless the client first establishes a valid `spiped` session.

- **Adds symmetric-key access control before SSH**  
  Only clients that already possess the shared `spiped` key can even *reach* `sshd`. SSH authentication remains in place, but protected by an extra security layer.

- **Prevents protocol fingerprinting**  
  Attackers cannot detect that SSH is running at all, reducing exposure to targeted SSH vulnerabilities.

- **Simple, low-overhead protection**  
  `spiped` uses fast symmetric cryptography and introduces minimal latency.

- **Defense in depth**  
  Attackers must bypass both the spiped key AND SSH authentication.

---

## How it works

`spiped` runs as a secure encrypted conduit between a *source address* (a public port) and a *target address* (typically `127.0.0.1:22`). All traffic is encrypted, authenticated, and verified before reaching `sshd`.

### Flow:

1. **Client initiates a spiped session** using the shared key.  
2. **Key exchange** occurs (Diffie-Hellman + HMAC-SHA256).  
3. **Traffic is encrypted** with AES-256-CTR + HMAC-SHA256.  
4. **Verified traffic** is forwarded to `sshd`.  
5. SSH authentication happens normally afterward.

### ASCII Diagram

```text
                ┌────────────────────────────────────────┐
                │               CLIENT                   │
                │                                        │
                │   ssh → spiped (client-side)           │
                └───────────────┬────────────────────────┘
                                │
                                │  Encrypted + Authenticated
                                │  (AES-256-CTR + HMAC-SHA256)
                                ▼
                     ┌────────────────────┐
                     │   NETWORK / WAN    │
                     └───────────┬────────┘
                                 │
                                 ▼
                ┌────────────────────────────────────────┐
                │               SERVER                   │
                │                                        │
                │  spiped (server-side)  →  sshd         │
                │     │ decrypt + verify     │           │
                │     └──────────────────────┘           │
                └────────────────────────────────────────┘
```

---

## Installation & setup (systemd)

This example uses:

- spiped listening on **0.0.0.0:8022**
- sshd listening on **127.0.0.1:22**
- key stored at **/etc/ssh/spiped-secret**

### 1. Generate the shared key

```bash
sudo dd if=/dev/urandom bs=32 count=1 of=/etc/ssh/spiped-secret
sudo chmod 600 /etc/ssh/spiped-secret
sudo chown root:root /etc/ssh/spiped-secret
```

### 2. Create the systemd service

Place this at: `/etc/systemd/system/spiped-ssh.service`

```ini
[Unit]
Description=Spiped for SSH
After=network.target

[Service]
Type=simple
ExecStart=/usr/bin/spiped -F -d -s '[0.0.0.0]:8022' -t '[127.0.0.1]:22' -k /etc/ssh/spiped-secret
Restart=always

[Install]
WantedBy=multi-user.target
```

Reload systemd:

```bash
sudo systemctl daemon-reload
```

### 3. Enable at boot

```bash
sudo systemctl enable spiped-ssh.service
```

### 4. Start the service

```bash
sudo systemctl start spiped-ssh.service
```

### 5. Stop / restart / disable

```bash
sudo systemctl stop spiped-ssh.service
sudo systemctl restart spiped-ssh.service
sudo systemctl disable spiped-ssh.service
```

---

## Connecting to a spiped-protected SSH server (client setup via ProxyCommand)

You can configure SSH to automatically launch `spiped` using the `ProxyCommand` directive. SSH starts spiped when connecting, pipes traffic through it, and closes it afterward.

### 1. Install spiped on the client

```bash
# Debian/Ubuntu
sudo apt install spiped

# Fedora/RHEL
sudo dnf install spiped

# macOS
brew install spiped
```

### 2. Copy the shared key to the client

Store it safely:

```
~/.config/spiped/ssh-secret
```

Fix permissions:

```bash
chmod 600 ~/.config/spiped/ssh-secret
```

### 3. Add SSH config entry

Edit `~/.ssh/config`:

```
Host my-spiped-server
    HostName 127.0.0.1
    User USERNAME
    Port 22
    ProxyCommand spiped -e -t '[SERVER_PUBLIC_IP]:8022' -k ~/.config/spiped/ssh-secret -F -o 2>/dev/null
```

### What this does

- SSH launches spiped automatically.
- spiped connects to the server’s encrypted spiped listener.
- SSH traffic flows through the secure tunnel.
- spiped terminates when the SSH session closes.

### 4. Connect normally

```bash
ssh my-spiped-server
```

### Optional: shorter alias

```
Host prod
    HostName 127.0.0.1
    User USERNAME
    ProxyCommand spiped -e -t '[SERVER_PUBLIC_IP]:8022' -k ~/.config/spiped/ssh-secret -F -o 2>/dev/null
```

Then:

```bash
ssh prod
```

---

With this setup, your SSH daemon is fully shielded behind spiped, dramatically reducing exposure while preserving your normal SSH workflow.
